//
//  BaseMessageNode.swift
//  TestTexture
//
//  Created by Aleksandr Miaots on 14.01.2021.
//

import AsyncDisplayKit

struct BaseMessageModel {
    
    let isOutgoing: Bool
    let senderName: String
    let dateString: String
    let text: String
}

final class ReplyMessageNode: ASDisplayNode {
    
    private let textNode = ASTextNode()
    private let senderNameNode = ASTextNode()
    private let dateNode = ASTextNode()
    private let lineNode = ASDisplayNode()
    
    init(with model: BaseMessageModel) {
        let textColor: UIColor = model.isOutgoing ? .white : .label
        let dateColor: UIColor = model.isOutgoing ? .init(white: 1, alpha: 0.6) : .systemGray
        senderNameNode.attributedText = NSAttributedString(string: model.senderName,
                                                           attributes: [.font: UIFont.boldSystemFont(ofSize: 14),
                                                                        .foregroundColor: textColor])
        dateNode.attributedText = NSAttributedString(string: model.dateString,
                                                     attributes: [.font: UIFont.systemFont(ofSize: 13),
                                                                  .foregroundColor: dateColor])
        textNode.attributedText = NSAttributedString(string: model.text,
                                                     attributes: [.font: UIFont.systemFont(ofSize: 16),
                                                                  .foregroundColor: textColor])
        lineNode.backgroundColor = textColor
        super.init()
        addSubnodes()
    }
    
    override func layoutSpecThatFits(_ constrainedSize: ASSizeRange) -> ASLayoutSpec {
        let titleStack = ASStackLayoutSpec(direction: .horizontal,
                                           spacing: 8,
                                           justifyContent: .start,
                                           alignItems: .center,
                                           children: [senderNameNode, dateNode])
        titleStack.style.flexShrink = 1.0
        titleStack.style.flexGrow = 1.0
        let messageStack = ASStackLayoutSpec(direction: .vertical,
                                            spacing: 8,
                                            justifyContent: .spaceBetween,
                                            alignItems: .stretch,
                                            children: [titleStack, textNode])
        messageStack.style.flexShrink = 1.0
        lineNode.style.preferredLayoutSize = .init(width: ASDimensionMake(2),
                                                   height: ASDimensionMakeWithFraction(1))
        return ASStackLayoutSpec(direction: .horizontal,
                                 spacing: 8,
                                 justifyContent: .start,
                                 alignItems: .stretch,
                                 children: [lineNode, messageStack])
    }
}

private extension ReplyMessageNode {
    
    func addSubnodes() {
        addSubnode(senderNameNode)
        addSubnode(dateNode)
        addSubnode(textNode)
        addSubnode(lineNode)
    }
}
