//
//  FileMessageNode.swift
//  TestTexture
//
//  Created by Aleksandr Miaots on 14.01.2021.
//

import AsyncDisplayKit

final class FileMessageNode: ASDisplayNode {
    
    private let fileNameNode = ASTextNode()
    private let progressView = ASDisplayNode()
}

// MARK: - DownloadProgressNode
class DownloadProgressNode: ASDisplayNode {
    
    enum State {
        
        case notDownloaded
        case progress(Double)
        case complete
        case error
    }
    
    var notDownloadedIcon: UIImage? = .init() {
        didSet {
            updateImageNode()
        }
    }
    
    var completeIcon: UIImage? = .init() {
        didSet {
            updateImageNode()
        }
    }
    
    var errorIcon: UIImage? = .init() {
        didSet {
            updateImageNode()
        }
    }
    
    var notDownloadedTintColor: UIColor = .blue {
        didSet {
            updateImageNode()
        }
    }
    
    var progressTintColor: UIColor = .blue {
        didSet {
            setNeedsDisplay()
        }
    }
    
    var completeTintColor: UIColor = .blue {
        didSet {
            updateImageNode()
        }
    }
    
    var errorTintColor: UIColor = .blue {
        didSet {
            updateImageNode()
        }
    }
    
    var downloadState: State = .notDownloaded {
        didSet {
            updateState()
        }
    }
    
    private let rectStopLayer = CALayer()
    
    private let progressCircleLayer: CAShapeLayer = {
        let layer = CAShapeLayer()
        layer.fillColor = UIColor.clear.cgColor
        layer.lineCap = .round
        layer.lineWidth = 2
        layer.strokeColor = UIColor.green.cgColor
        layer.transform = CATransform3DMakeRotation(-.pi/2, 0, 0, 1)
        return layer
    }()
    
    private let imageNode = ASImageNode()
    
    override init() {
        super.init()
        imageNode.isLayerBacked = true
        isLayerBacked = false
        addSubnode(imageNode)
    }
    
    override func didLoad() {
        super.didLoad()
        setupLayers()
    }
    
    deinit {
        imageNode.removeFromSupernode()
    }
    
    override func layout() {
        super.layout()
        layoutLayers()
    }
    
    override func layoutSpecThatFits(_ constrainedSize: ASSizeRange) -> ASLayoutSpec {
        return ASWrapperLayoutSpec(layoutElement: imageNode)
    }
}

private extension DownloadProgressNode {
    
    func setupLayers() {
        layer.addSublayer(rectStopLayer)
        layer.addSublayer(progressCircleLayer)
        progressCircleLayer.strokeStart = 0.0
        progressCircleLayer.strokeEnd = 0.0
        rectStopLayer.opacity = 0
        progressCircleLayer.opacity = 0
    }
    
    func layoutLayers() {
        rectStopLayer.backgroundColor = progressTintColor.cgColor
        progressCircleLayer.strokeColor = progressTintColor.cgColor
        rectStopLayer.frame = bounds
        progressCircleLayer.frame = bounds
        let sideSize = min(bounds.height, bounds.width) / 3
        let rectPath = CGRect(x: bounds.midX - sideSize / 2,
                              y: bounds.midY - sideSize / 2,
                              width: sideSize,
                              height: sideSize)
        rectStopLayer.frame = rectPath
        rectStopLayer.cornerRadius = sideSize * 0.2
        progressCircleLayer.path = UIBezierPath(ovalIn: bounds.insetBy(dx: 2, dy: 2)).cgPath
    }
    
    func updateState() {
//        DispatchQueue.main.async {
            self.updateViews()
//        }
    }
    
    func updateImageNode() {
        var imageIsHidden = false
        var imageTintColor: UIColor = .clear
        switch downloadState {
        case .notDownloaded:
            imageNode.image = notDownloadedIcon
            imageTintColor = notDownloadedTintColor
        case .progress:
            imageIsHidden = true
        case .complete:
            imageNode.image = completeIcon
            imageTintColor = completeTintColor
        case .error:
            imageNode.image = errorIcon
            imageTintColor = errorTintColor
        }
        imageNode.imageModificationBlock = ASImageNodeTintColorModificationBlock(imageTintColor)
        imageNode.isHidden = imageIsHidden
        imageNode.setNeedsDisplay()
    }
    
    func updateViews() {
        updateImageNode()
        var isVisibleProgress = false
        switch downloadState {
        case .progress(let value):
            isVisibleProgress = true
            progressCircleLayer.strokeEnd = CGFloat(value)
        case .complete:
            progressCircleLayer.strokeEnd = 1
        case .error, .notDownloaded:
            progressCircleLayer.strokeEnd = 0
        }
        CATransaction.begin()
        CATransaction.setDisableActions(true)
        progressCircleLayer.opacity = isVisibleProgress ? 1 : 0
        rectStopLayer.opacity = isVisibleProgress ? 1 : 0
        CATransaction.commit()
    }
}
