//
//  Layouts.swift
//  TestTexture
//
//  Created by Aleksandr Miaots on 29.01.2021.
//

import UIKit

class RotatableCollectionViewFlowLayout: UICollectionViewFlowLayout {

    private var focusedIndexPath: IndexPath?

    override func prepare(forAnimatedBoundsChange oldBounds: CGRect) {
        super.prepare(forAnimatedBoundsChange: oldBounds)
        focusedIndexPath = collectionView?.indexPathsForVisibleItems.first
    }

    override func targetContentOffset(forProposedContentOffset proposedContentOffset: CGPoint) -> CGPoint {
        guard let indexPath = focusedIndexPath,
              let attributes = layoutAttributesForItem(at: indexPath),
              let collectionView = collectionView else {
                return super.targetContentOffset(forProposedContentOffset: proposedContentOffset)
        }
        return CGPoint(x: attributes.frame.origin.x - collectionView.contentInset.left,
                       y: attributes.frame.origin.y - collectionView.contentInset.top)
    }

    override func finalizeAnimatedBoundsChange() {
        super.finalizeAnimatedBoundsChange()
        focusedIndexPath = nil
    }
}

class ChatCollectionViewFlowLayout: UICollectionViewFlowLayout {
    
    private var topMostVisibleItem    =  Int.max
    private var bottomMostVisibleItem = -Int.max
    
    private var offset: CGFloat = 0.0
    private var visibleAttributes: [UICollectionViewLayoutAttributes]?
    
    private var isInsertingItemsToTop    = false
    private var isInsertingItemsToBottom = false
    
    private var focusedIndexPath: IndexPath?

    override func prepare(forAnimatedBoundsChange oldBounds: CGRect) {
        super.prepare(forAnimatedBoundsChange: oldBounds)
        focusedIndexPath = collectionView?.indexPathsForVisibleItems.first
    }

    override func targetContentOffset(forProposedContentOffset proposedContentOffset: CGPoint) -> CGPoint {
        guard let indexPath = focusedIndexPath,
              let attributes = layoutAttributesForItem(at: indexPath),
              let collectionView = collectionView else {
                return super.targetContentOffset(forProposedContentOffset: proposedContentOffset)
        }
        return CGPoint(x: attributes.frame.origin.x - collectionView.contentInset.left,
                       y: attributes.frame.origin.y - collectionView.contentInset.top)
    }

    override func finalizeAnimatedBoundsChange() {
        super.finalizeAnimatedBoundsChange()
        focusedIndexPath = nil
    }
    
    override func layoutAttributesForElements(in rect: CGRect) -> [UICollectionViewLayoutAttributes]? {
        
        // Reset each time all values to recalculate them
        // ════════════════════════════════════════════════════════════
        
        // Get layout attributes of all items
        visibleAttributes = super.layoutAttributesForElements(in: rect)
        
        // Erase offset
        offset = 0.0
        
        // Reset inserting flags
        isInsertingItemsToTop    = false
        isInsertingItemsToBottom = false
        
        return visibleAttributes
    }
    
    override func prepare(forCollectionViewUpdates updateItems: [UICollectionViewUpdateItem]) {
        
        // Check where new items get inserted
        // ════════════════════════════════════════════════════════════
        
        // Get collection view and layout attributes as non-optional object
        guard let collectionView = self.collectionView       else { return }
        guard let visibleAttributes = self.visibleAttributes else { return }

        // Find top and bottom most visible item
        // ────────────────────────────────────────────────────────────
        
        bottomMostVisibleItem = -Int.max
        topMostVisibleItem    =  Int.max

        let container = CGRect(x: collectionView.contentOffset.x,
                               y: collectionView.contentOffset.y,
                               width:  collectionView.frame.size.width,
                               height: (collectionView.frame.size.height - (collectionView.contentInset.top + collectionView.contentInset.bottom)))
        
        for attributes in visibleAttributes {
            
            // Check if cell frame is inside container frame
            if attributes.frame.intersects(container) {
                let item = attributes.indexPath.item
                if item < topMostVisibleItem    { topMostVisibleItem    = item }
                if item > bottomMostVisibleItem { bottomMostVisibleItem = item }
            }
        }
        
        
        // Call super after first calculations
        super.prepare(forCollectionViewUpdates: updateItems)
        
        // Calculate offset of inserting items
        // ────────────────────────────────────────────────────────────
        
        var willInsertItemsToTop    = false
        var willInsertItemsToBottom = false
        
        // Iterate over all new items and add their height if they go inserted
        for updateItem in updateItems {
            switch updateItem.updateAction {
            case .insert:
                if topMostVisibleItem + updateItems.count > updateItem.indexPathAfterUpdate!.item {
                    if let newAttributes = self.layoutAttributesForItem(at: updateItem.indexPathAfterUpdate!) {
                        
                        offset += (newAttributes.size.height + self.minimumLineSpacing)
                        willInsertItemsToTop = true
                    }
                    
                } else if bottomMostVisibleItem <= updateItem.indexPathAfterUpdate!.item {
                    if let newAttributes = self.layoutAttributesForItem(at: updateItem.indexPathAfterUpdate!) {
                        
                        offset += (newAttributes.size.height + self.minimumLineSpacing)
                        willInsertItemsToBottom = true
                    }
                }
                
            case.delete:
                // TODO: Handle removal of items
                break
                
            default:
                break
            }
        }
        
        // Pass on information if items need more than one screen
        // ────────────────────────────────────────────────────────────
        
        // Just continue if one flag is set
        if willInsertItemsToTop || willInsertItemsToBottom {
            
            // Get heights without top and bottom
            let collectionViewContentHeight = collectionView.contentSize.height
            let collectionViewFrameHeight   = collectionView.frame.size.height - (collectionView.contentInset.top + collectionView.contentInset.bottom)
            
            // Continue only if the new content is higher then the frame
            // If it is not the case the collection view can display all cells on one screen
            if collectionViewContentHeight + offset > collectionViewFrameHeight {
                
                if willInsertItemsToTop {
                    CATransaction.begin()
                    CATransaction.setDisableActions(true)
                    isInsertingItemsToTop = true
                    
                } else if willInsertItemsToBottom {
                    isInsertingItemsToBottom = true
                }
            }
        }
    }
    
    override func finalizeCollectionViewUpdates() {
        
        // Set final content offset with animation or not
        // ════════════════════════════════════════════════════════════
        
        // Get collection view as non-optional object
        guard let collectionView = self.collectionView else { return }
        
        if isInsertingItemsToTop {
            
            // Calculate new content offset
            let newContentOffset = CGPoint(x: collectionView.contentOffset.x,
                                           y: collectionView.contentOffset.y + offset)
            
            // Set new content offset without animation
            collectionView.contentOffset = newContentOffset
            
            // Commit/end transaction
            CATransaction.commit()
            
        } else if isInsertingItemsToBottom {

            // Calculate new content offset
            // Always scroll to bottom
            let newContentOffset = CGPoint(x: collectionView.contentOffset.x,
                                           y: collectionView.contentSize.height + offset - collectionView.frame.size.height + collectionView.contentInset.bottom)
            
            // Set new content offset with animation
            collectionView.setContentOffset(newContentOffset, animated: true)
        }
    }
}

class ChatFlowLayoutExample: UICollectionViewFlowLayout {
    
    private var topVisibleItem =  Int.max
    private var bottomVisibleItem = -Int.max
    private var offset: CGFloat = 0.0
    private var visibleAttributes: [UICollectionViewLayoutAttributes]?
    private var isPrepend: Bool = false
    
    private var focusedIndexPath: IndexPath?

    override func prepare(forAnimatedBoundsChange oldBounds: CGRect) {
        super.prepare(forAnimatedBoundsChange: oldBounds)
        focusedIndexPath = collectionView?.indexPathsForVisibleItems.first
    }

    override func targetContentOffset(forProposedContentOffset proposedContentOffset: CGPoint) -> CGPoint {
        guard let indexPath = focusedIndexPath,
              let attributes = layoutAttributesForItem(at: indexPath),
              let collectionView = collectionView else {
                return super.targetContentOffset(forProposedContentOffset: proposedContentOffset)
        }
        return CGPoint(x: attributes.frame.origin.x - collectionView.contentInset.left,
                       y: attributes.frame.origin.y - collectionView.contentInset.top)
    }

    override func finalizeAnimatedBoundsChange() {
        super.finalizeAnimatedBoundsChange()
        focusedIndexPath = nil
    }
    
    override open func layoutAttributesForElements(in rect: CGRect) -> [UICollectionViewLayoutAttributes]? {
        // Reset offset and prepend scope
        visibleAttributes = super.layoutAttributesForElements(in: rect)
        offset = 0.0
        isPrepend = false
        return visibleAttributes
    }
    
    override open func prepare(forCollectionViewUpdates updateItems: [UICollectionViewUpdateItem]) {
        guard let collectionView = self.collectionView else { return }
        guard let visibleAttributes = self.visibleAttributes else { return }
      
        // Calculate Bottom and Top Visible Item Count
        bottomVisibleItem = -Int.max
        topVisibleItem = Int.max
        var containerHeight: CGFloat = collectionView.frame.size.height
        containerHeight -= collectionView.contentInset.top
        containerHeight -= collectionView.contentInset.bottom
        let container = CGRect(x: collectionView.contentOffset.x,
                               y: collectionView.contentOffset.y,
                               width: collectionView.frame.size.width,
                               height: containerHeight)
        for attributes in visibleAttributes {
            if attributes.frame.intersects(container) {
                let item = attributes.indexPath.item
                
                if item < topVisibleItem {
                    topVisibleItem = item
                }
                
                if item > bottomVisibleItem {
                    bottomVisibleItem = item
                }
            }
        }
        
        super.prepare(forCollectionViewUpdates: updateItems)
        // Check: Initial Load or Load More
        let isInitialLoading: Bool = bottomVisibleItem + topVisibleItem == 0
        
        // Chack: Pre-Append or Append
        if updateItems.first?.indexPathAfterUpdate?.item ?? -1 == 0,
            updateItems.first?.updateAction == .insert,
            !isInitialLoading {
            self.isPrepend = true
        } else {
            return
        }
        
        // Calculate Offset
        offset = updateItems.filter { $0.updateAction == .insert }
            .compactMap { $0.indexPathAfterUpdate }
            .filter { topVisibleItem + updateItems.count > $0.item }
            .compactMap { self.layoutAttributesForItem(at: $0) }
            .map { $0.size.height + self.minimumLineSpacing }
            .reduce(0.0, { $0 + $1 })
        
        let contentHeight = collectionView.contentSize.height
        var frameHeight = collectionView.frame.size.height
        frameHeight -= collectionView.contentInset.top
        frameHeight -= collectionView.contentInset.bottom
        guard contentHeight + offset > frameHeight else {
            // Exception
            self.isPrepend = false
            return
        }
        CATransaction.begin()
        CATransaction.setDisableActions(true)
    }
    
    override open func finalizeCollectionViewUpdates() {
        guard let collectionView = self.collectionView, isPrepend else { return }
        // Adjust Content Offset
        let newContentOffset = CGPoint(x: collectionView.contentOffset.x,
                                       y: collectionView.contentOffset.y + self.offset)
        collectionView.contentOffset = newContentOffset
        CATransaction.commit()
    }
}

